<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>System Update Manager</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://maxcdn.bootstrapcdn.com/bootstrap/3.3.7/css/bootstrap.min.css" rel="stylesheet">
    <style>
        .update-container {
            max-width: 800px;
            margin: 50px auto;
            padding: 20px;
        }
        .status-panel {
            margin-bottom: 30px;
        }
        .progress-container {
            margin: 20px 0;
        }
        .log-container {
            max-height: 400px;
            overflow-y: auto;
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 5px;
            padding: 15px;
        }
        .log-entry {
            margin-bottom: 5px;
            font-family: monospace;
            font-size: 12px;
        }
        .loading-spinner {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid #f3f3f3;
            border-top: 3px solid #3498db;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin-right: 10px;
        }
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="update-container">
            <div class="page-header">
                <h1><i class="fa fa-refresh"></i> System Update Manager</h1>
                <p class="lead">Manage system updates and database migrations</p>
            </div>

            <!-- Status Panel -->
            <div class="panel panel-info status-panel">
                <div class="panel-heading">
                    <h3 class="panel-title"><i class="fa fa-info-circle"></i> System Status</h3>
                </div>
                <div class="panel-body">
                    <div id="status-content">
                        <div class="text-center">
                            <div class="loading-spinner" id="checking-spinner"></div>
                            <p id="status-message">Checking system status...</p>
                        </div>
                        <div id="version-info" style="display: none;">
                            <div class="row">
                                <div class="col-md-6">
                                    <strong>Current Version:</strong> <span id="current-version">-</span>
                                </div>
                                <div class="col-md-6">
                                    <strong>Latest Version:</strong> <span id="latest-version">-</span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Update Options -->
            <div class="panel panel-default">
                <div class="panel-heading">
                    <h3 class="panel-title"><i class="fa fa-cog"></i> Update Options</h3>
                </div>
                <div class="panel-body">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="checkbox">
                                <label>
                                    <input type="checkbox" id="backup-files" checked> Create backup before updating
                                </label>
                            </div>
                            <div class="checkbox">
                                <label>
                                    <input type="checkbox" id="update-database" checked> Update database schema
                                </label>
                            </div>
                            <div class="checkbox">
                                <label>
                                    <input type="checkbox" id="preserve-config" checked> Preserve existing configuration
                                </label>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <button class="btn btn-primary btn-block" id="check-update-btn">
                                <i class="fa fa-search"></i> Check for Updates
                            </button>
                            <button class="btn btn-warning btn-block" id="update-btn" style="display: none;">
                                <i class="fa fa-refresh"></i> Update System
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Progress Panel -->
            <div class="panel panel-default" id="progress-panel" style="display: none;">
                <div class="panel-heading">
                    <h3 class="panel-title"><i class="fa fa-tasks"></i> Update Progress</h3>
                </div>
                <div class="panel-body">
                    <div class="progress">
                        <div class="progress-bar progress-bar-striped active" role="progressbar" 
                             style="width: 0%" id="progress-bar">
                            <span id="progress-text">0%</span>
                        </div>
                    </div>
                    <div id="progress-message" class="text-center" style="margin-top: 10px;">
                        Preparing update...
                    </div>
                </div>
            </div>

            <!-- Update Log -->
            <div class="panel panel-default">
                <div class="panel-heading">
                    <h3 class="panel-title">
                        <i class="fa fa-list"></i> Update Log
                        <button class="btn btn-xs btn-default pull-right" id="refresh-log-btn">
                            <i class="fa fa-refresh"></i> Refresh
                        </button>
                    </h3>
                </div>
                <div class="panel-body">
                    <div class="log-container" id="log-container">
                        <div class="text-center text-muted">
                            <i class="fa fa-info-circle"></i> No update logs available
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://maxcdn.bootstrapcdn.com/bootstrap/3.3.7/js/bootstrap.min.js"></script>
    <script>
        $(document).ready(function() {
            // Check system status on page load
            checkSystemStatus();
            
            // Event handlers
            $('#check-update-btn').click(function() {
                checkSystemStatus();
            });
            
            $('#update-btn').click(function() {
                performUpdate();
            });
            
            $('#refresh-log-btn').click(function() {
                loadUpdateLog();
            });
        });

        function checkSystemStatus() {
            $('#checking-spinner').show();
            $('#status-message').text('Checking system status...');
            
            $.ajax({
                url: '<?= base_url('update_manager/check_update') ?>',
                type: 'GET',
                dataType: 'json',
                success: function(response) {
                    $('#checking-spinner').hide();
                    
                    if(response.success) {
                        $('#current-version').text(response.current_version);
                        $('#latest-version').text(response.target_version);
                        $('#version-info').show();
                        
                        if(response.needs_update) {
                            $('#status-message').html('<span class="text-warning"><i class="fa fa-exclamation-triangle"></i> Update available!</span>');
                            $('#update-btn').show();
                        } else {
                            $('#status-message').html('<span class="text-success"><i class="fa fa-check-circle"></i> System is up to date!</span>');
                            $('#update-btn').hide();
                        }
                    } else {
                        $('#status-message').html('<span class="text-danger"><i class="fa fa-exclamation-circle"></i> Error checking status: ' + response.message + '</span>');
                    }
                },
                error: function() {
                    $('#checking-spinner').hide();
                    $('#status-message').html('<span class="text-danger"><i class="fa fa-exclamation-circle"></i> Failed to check system status</span>');
                }
            });
        }

        function performUpdate() {
            $('#update-btn').prop('disabled', true).html('<span class="loading-spinner"></span> Updating...');
            $('#progress-panel').show();
            $('#progress-message').text('Starting update process...');
            
            var updateOptions = {
                backup_files: $('#backup-files').is(':checked'),
                update_database: $('#update-database').is(':checked'),
                preserve_config: $('#preserve-config').is(':checked')
            };
            
            $.ajax({
                url: '<?= base_url('update_manager/perform_update') ?>',
                type: 'POST',
                data: updateOptions,
                dataType: 'json',
                success: function(response) {
                    if(response.success) {
                        $('#progress-bar').css('width', '100%').removeClass('active');
                        $('#progress-text').text('100%');
                        $('#progress-message').html('<span class="text-success"><i class="fa fa-check-circle"></i> ' + response.message + '</span>');
                        $('#update-btn').hide();
                        
                        // Refresh status
                        setTimeout(function() {
                            checkSystemStatus();
                        }, 2000);
                    } else {
                        $('#progress-message').html('<span class="text-danger"><i class="fa fa-exclamation-circle"></i> ' + response.message + '</span>');
                        $('#update-btn').prop('disabled', false).html('<i class="fa fa-refresh"></i> Retry Update');
                    }
                },
                error: function() {
                    $('#progress-message').html('<span class="text-danger"><i class="fa fa-exclamation-circle"></i> Update failed due to network error</span>');
                    $('#update-btn').prop('disabled', false).html('<i class="fa fa-refresh"></i> Retry Update');
                }
            });
        }

        function loadUpdateLog() {
            $.ajax({
                url: '<?= base_url('update_manager/get_update_log') ?>',
                type: 'GET',
                dataType: 'json',
                success: function(response) {
                    var logContainer = $('#log-container');
                    
                    if(response.logs && response.logs.length > 0) {
                        var logHtml = '';
                        response.logs.forEach(function(log) {
                            logHtml += '<div class="log-entry">' + log + '</div>';
                        });
                        logContainer.html(logHtml);
                    } else {
                        logContainer.html('<div class="text-center text-muted"><i class="fa fa-info-circle"></i> No update logs available</div>');
                    }
                },
                error: function() {
                    $('#log-container').html('<div class="text-center text-danger"><i class="fa fa-exclamation-circle"></i> Failed to load update logs</div>');
                }
            });
        }
    </script>
</body>
</html>
