<?php
/**
 * Backup Cleanup Utility
 * 
 * This script removes old backup files to prevent disk space issues.
 * Run this script periodically to maintain a clean backup directory.
 */

// Configuration
$backup_directory = '../../backups/';
$retention_days = 30; // Keep backups for 30 days
$dry_run = false; // Set to true to see what would be deleted without actually deleting

// Function to recursively delete directories
function deleteDirectory($dir) {
    if (!is_dir($dir)) {
        return false;
    }
    
    $files = array_diff(scandir($dir), array('.', '..'));
    foreach ($files as $file) {
        $path = $dir . '/' . $file;
        if (is_dir($path)) {
            deleteDirectory($path);
        } else {
            unlink($path);
        }
    }
    return rmdir($dir);
}

// Function to get directory age in days
function getDirectoryAge($dir) {
    $stat = stat($dir);
    $created_time = $stat['mtime'];
    $current_time = time();
    return floor(($current_time - $created_time) / (60 * 60 * 24));
}

// Main cleanup process
function cleanupBackups($backup_dir, $retention_days, $dry_run = false) {
    if (!is_dir($backup_dir)) {
        echo "Backup directory does not exist: $backup_dir\n";
        return;
    }
    
    $deleted_count = 0;
    $total_size = 0;
    
    echo "Starting backup cleanup...\n";
    echo "Backup directory: $backup_dir\n";
    echo "Retention period: $retention_days days\n";
    echo "Mode: " . ($dry_run ? "DRY RUN (no files will be deleted)" : "LIVE RUN") . "\n\n";
    
    $directories = glob($backup_dir . '*', GLOB_ONLYDIR);
    
    foreach ($directories as $dir) {
        $dir_name = basename($dir);
        $age = getDirectoryAge($dir);
        
        // Check if directory name matches backup pattern (YYYY-MM-DD_HH-MM-SS)
        if (preg_match('/^\d{4}-\d{2}-\d{2}_\d{2}-\d{2}-\d{2}$/', $dir_name)) {
            if ($age > $retention_days) {
                $size = getDirectorySize($dir);
                $total_size += $size;
                
                echo "Found old backup: $dir_name (age: {$age} days, size: " . formatBytes($size) . ")\n";
                
                if (!$dry_run) {
                    if (deleteDirectory($dir)) {
                        echo "  ✓ Deleted successfully\n";
                        $deleted_count++;
                    } else {
                        echo "  ✗ Failed to delete\n";
                    }
                } else {
                    echo "  [DRY RUN] Would delete this directory\n";
                    $deleted_count++;
                }
            } else {
                echo "Keeping backup: $dir_name (age: {$age} days)\n";
            }
        } else {
            echo "Skipping non-backup directory: $dir_name\n";
        }
    }
    
    echo "\nCleanup Summary:\n";
    echo "Directories processed: " . count($directories) . "\n";
    echo "Directories " . ($dry_run ? "that would be" : "") . " deleted: $deleted_count\n";
    echo "Total space " . ($dry_run ? "that would be" : "") . " freed: " . formatBytes($total_size) . "\n";
}

// Function to calculate directory size
function getDirectorySize($dir) {
    $size = 0;
    if (is_dir($dir)) {
        $files = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($dir));
        foreach ($files as $file) {
            if ($file->isFile()) {
                $size += $file->getSize();
            }
        }
    }
    return $size;
}

// Function to format bytes in human readable format
function formatBytes($bytes, $precision = 2) {
    $units = array('B', 'KB', 'MB', 'GB', 'TB');
    
    for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
        $bytes /= 1024;
    }
    
    return round($bytes, $precision) . ' ' . $units[$i];
}

// Command line interface
if (php_sapi_name() === 'cli') {
    // Parse command line arguments
    $options = getopt('d:r:n', ['dry-run', 'help']);
    
    if (isset($options['help'])) {
        echo "Backup Cleanup Utility\n\n";
        echo "Usage: php cleanup_backups.php [options]\n\n";
        echo "Options:\n";
        echo "  -d <days>     Retention period in days (default: 30)\n";
        echo "  -r <path>     Backup directory path (default: ../../backups/)\n";
        echo "  -n, --dry-run Show what would be deleted without actually deleting\n";
        echo "  --help        Show this help message\n\n";
        echo "Examples:\n";
        echo "  php cleanup_backups.php                    # Clean up backups older than 30 days\n";
        echo "  php cleanup_backups.php -d 7               # Clean up backups older than 7 days\n";
        echo "  php cleanup_backups.php --dry-run          # Show what would be deleted\n";
        exit(0);
    }
    
    if (isset($options['d'])) {
        $retention_days = (int)$options['d'];
    }
    
    if (isset($options['r'])) {
        $backup_directory = $options['r'];
    }
    
    if (isset($options['n']) || isset($options['dry-run'])) {
        $dry_run = true;
    }
    
    cleanupBackups($backup_directory, $retention_days, $dry_run);
} else {
    // Web interface
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <title>Backup Cleanup Utility</title>
        <link href="https://maxcdn.bootstrapcdn.com/bootstrap/3.3.7/css/bootstrap.min.css" rel="stylesheet">
    </head>
    <body>
        <div class="container" style="margin-top: 50px;">
            <div class="row">
                <div class="col-md-8 col-md-offset-2">
                    <div class="panel panel-default">
                        <div class="panel-heading">
                            <h3 class="panel-title">Backup Cleanup Utility</h3>
                        </div>
                        <div class="panel-body">
                            <form method="POST">
                                <div class="form-group">
                                    <label for="retention_days">Retention Period (days):</label>
                                    <input type="number" class="form-control" id="retention_days" name="retention_days" 
                                           value="<?= $retention_days ?>" min="1" max="365">
                                </div>
                                <div class="form-group">
                                    <div class="checkbox">
                                        <label>
                                            <input type="checkbox" name="dry_run" value="1" <?= $dry_run ? 'checked' : '' ?>>
                                            Dry run (show what would be deleted without actually deleting)
                                        </label>
                                    </div>
                                </div>
                                <button type="submit" class="btn btn-primary">Run Cleanup</button>
                            </form>
                            
                            <?php if ($_SERVER['REQUEST_METHOD'] === 'POST'): ?>
                                <hr>
                                <h4>Cleanup Results:</h4>
                                <pre style="background: #f5f5f5; padding: 15px; border-radius: 4px;">
<?php
$retention_days = (int)($_POST['retention_days'] ?? $retention_days);
$dry_run = isset($_POST['dry_run']);
cleanupBackups($backup_directory, $retention_days, $dry_run);
?>
                                </pre>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </body>
    </html>
    <?php
}
?>

